<?php
/*
Plugin Name: Woocommerce integration with Kommo/amoCRM
Description: Flexible integration of Woocommerce with amoCRM/Kommo
Version: 1.0
Requires Plugins: woocommerce
Author: crmapp.store
Author URI: https://crmapp.store
*/

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('wc_amocrm')) {
    class wc_amocrm {
        
        const PLUGIN_NAME = 'wc_amocrm';
		const PAGE_ID = 'wc_amocrm';
		const LANGS = ['ru', 'en', 'pt', 'es'];
		const CURL_TIMEOUT = 10; 
		const HOST = 'https://devamo.ru/woocommerce/ajax';
		const PROXY = 'https://functions.yandexcloud.net/d4ek9e15tnll0dcuakaq';
		const CURRENT_VERSION = '20250704';

		private static $locale;
		private static $lang_;
		private $lang;

        // Конструктор
        public function __construct() {
			$get_locale = get_locale();
			$locale_arr = explode('_', $get_locale);
			$locale = $locale_arr[0];

			if(!in_array($locale, self::LANGS)){
				$locale = 'en';
			}
			self::$locale = $locale;
			$langs = require_once __DIR__ . '/langs.php';
			$this->lang = $langs[$locale];
			self::$lang_ = $this->lang;

            add_action('init', [$this, 'save_utm_to_cookie']);


			add_filter('query_vars', [$this, 'add_query_var']);
			add_action('parse_request', [$this, 'handle_cron_endpoint']);

			
            add_action('admin_menu', [$this, 'add_admin_menu']);

			add_action('woocommerce_checkout_order_processed', [$this, 'add_task'],  99, 1);
			add_action('woocommerce_store_api_checkout_order_processed', [$this, 'add_task'], 99, 1);

			add_action('delay_send_order_' . self::PLUGIN_NAME, [$this, 'send_order'], 10, 1); //задержка перед отправкой

			add_action('woocommerce_order_status_changed', [$this, 'order_status_changed'], 10, 4 );

			add_action('wp_ajax_export_'.self::PLUGIN_NAME, [$this, 'export']);
			add_action('wp_ajax_orders_'.self::PLUGIN_NAME, [$this, 'orders']);
			add_action('wp_ajax_nopriv_changeOrderStatus_'.self::PLUGIN_NAME, [$this, 'change_order_status']);

            add_action('admin_enqueue_scripts', [$this, 'add_scripts'], 11);
			add_action('wp_print_footer_scripts', [$this, 'add_footer']);
			add_filter('all_plugins', [$this, 'translate_name_description']);
        }

		public function add_query_var($vars) {
			$vars[] = 'wc_amocrm_cron';
			return $vars;
		}

		public function handle_cron_endpoint($wp) {
			if (isset($wp->query_vars['wc_amocrm_cron'])) {
				error_log('handle_cron_endpoint запущен');

				global $wpdb;
				$tasks = $wpdb->get_results(
					$wpdb->prepare('
						SELECT * FROM `'.$wpdb->prefix.'woocommerce_delayed_orders` 
						WHERE (pushed_at + delay) <= %d
						AND attempt = 0
					', 
					[time()]
					)
				);
				if($tasks){
					foreach($tasks as $task){
						$this->send_order($task->order_id);
						$wpdb->update($wpdb->prefix.'woocommerce_delayed_orders', ['attempt' => 1], ['order_id' => $task->order_id]);
					}
				}
			}
		}

		public function add_task($order_id){
			if(is_object($order_id)){ // если блочный интерфейс оформления заказа
				$order_id = $order_id->get_id();
			}

			$settings = get_option(self::PLUGIN_NAME . '_settings');
			$delay = isset($settings['delay_send_amo']) ? (int)$settings['delay_send_amo'] : 0; // По умолчанию 2 часа
			
			// Если задержка = 0, отправляем сразу
			if (!$delay) {
				$this->send_order($order_id);
				return;
			}
			
			// Сохраняем заказ во временном хранилище
			global $wpdb;
			$wpdb->insert(
				$wpdb->prefix . 'woocommerce_delayed_orders',
				[
					'order_id'    => $order_id,
					'pushed_at'   => time(),
					'delay'  	  => $delay,
					'attempt' 	  => 0,
				],
				['%d', '%d', '%d', '%d']
			);
		}

		public function order_status_changed( $order_id, $status_old, $status_new, $that ){
			$settings = get_option(self::PLUGIN_NAME.'_settings');
			if( !isset($settings['wc_statuses']['wc-'.$status_new]) and (int)$settings['wc_statuses']['wc-'.$status_new] ){
				return false;
			}
			$result = self::send_data([
				'action' => 'newLeadStatus',
				'url' => $_SERVER['HTTP_HOST'],
				'status_id' => (int)$settings['wc_statuses']['wc-'.$status_new],
				'order_id' => $order_id,
			]);			
		}

		public function change_order_status(){
			$post = $_POST;
			$settings = get_option(self::PLUGIN_NAME.'_settings');
			
			if(!$settings['plugin_status']){
				echo json_encode(['status' => 0, 'message' => 'Плагин отключен']);
				wp_die();
			}

			$order_id = $post['order_id'];
			$data = ['status' => 0];
			$wc_statuses = $settings['wc_statuses'];
			$wc_order_status_id = array_search($post['order_status_id'], $wc_statuses);
			
			if($wc_order_status_id !== false){
				$order_data = new WC_Order($order_id);
				$data = ['status' => $order_data->update_status($wc_order_status_id)];
			}
			echo json_encode($data);
			wp_die();
		}

		// отправляет в амо
		public function order_data($order_id = 0, $write_utm = false){
			$settings = get_option(self::PLUGIN_NAME.'_settings');
			if(!(int)$settings['plugin_status']){
				return false;
			}

			global $wpdb;

			$order_info = $this->order($order_id);
			$order_fields = array_keys($order_info);
			$comparison = isset($settings['comparison']) ? $settings['comparison'] : [];
			$amo_settings = get_option(self::PLUGIN_NAME.'_amocrm');
			$responsible_user_id = $this->responsible_user_id();
			$pipeline_id = isset($settings['pipeline_id']) ? (int)$settings['pipeline_id'] : 0;
			$status_id = isset($settings['wc_statuses']['wc-'.$order_info['status_label']]) ? (int)$settings['wc_statuses']['wc-'.$order_info['status_label']] : 0;

			$entities = [
				'lead' => [ // начальные данные для сделки
					'pipeline_id' => $pipeline_id,
					'status_id' => $status_id,
					'responsible_user_id' => $responsible_user_id,
				]
			];

			if((int)$settings['note_status']){ // примечание
				$entities['note'] = $settings['note_template'];
			}

			// пишем utm в бд
			if($write_utm and isset($amo_settings['utm_fields'])){
				foreach($amo_settings['utm_fields'] as $utm_field_id => $utm){
					if( !isset($_COOKIE[$utm]) or $_COOKIE[$utm] === '' ){
						continue;
					}

					$wpdb->insert( $wpdb->prefix.'woocommerce_order_utms', 
						[
							'order_id' => $order_id,
							'name' => $utm,
							'value' => $_COOKIE[$utm],
						],
						[ '%d', '%s', '%s' ] 
					);
				}
			}

			// читаем utm
			if( isset($amo_settings['utm_fields']) ){
				$utms = $wpdb->get_results( 'SELECT name, value FROM '. $wpdb->prefix .'woocommerce_order_utms WHERE order_id = '.(int)$order_id );
				$utms_data = [];
				if($utms){
					foreach($utms as $utm){
						$utms_data[$utm->name] = $utm->value;
					}
				}
				foreach($amo_settings['utm_fields'] as $utm_field_id => $utm){
					if( !isset($utms_data[$utm]) ){
						continue;
					}
					$utm_value = $utms_data[$utm];
					$entities['lead']['custom_fields_values'][] = [
						'field_id' => (int)$utm_field_id,
						'values' => [
							[
								'value' => $utm_value
							]
						]
					];         
				}
			}

			// заменим теги на значения в названии сделки
			if(isset($settings['lead_name']) and $settings['lead_name'] !== ''){
				$lead_name = $settings['lead_name'];
				foreach($order_fields as $order_field){
					if(is_array($order_info[$order_field])){
						continue;
					}
					$order_value = $order_info[$order_field];
					if($order_value === null){
						$order_value = '';
					}
					$lead_name = str_replace('{{'.$order_field.'}}', $order_value, $lead_name);
				}
				$entities['lead']['name'] = $lead_name;
			}

			// запишем в сущности значения полей из вкладки конструктор
			if($comparison){
				foreach($comparison as $order_field => $amo_field){
					
					if($amo_field !== '0'){
						$field_keys = explode('-', $amo_field);
						$entity = $field_keys['0']; // сущность
						$field_id = $field_keys['1']; // ID поля в amoCRM
						$order_value = $order_info[$order_field];

						$field_type = 'text';

						if(isset($amo_settings['field_types'][$entity][$field_id])){
							$field_type = $amo_settings['field_types'][$entity][$field_id];
						}
						
						if( in_array($field_type, ['date_time', 'birthday', 'date']) ){
							$order_value = strtotime($order_value);
						}elseif( in_array($field_type, ['checkbox']) ){
							$order_value = $order_value ? true : false;
						}else{
							$order_value = (string)$order_value;
						}

						if($order_value === '' or $order_value === null){
							continue;
						}

						if($field_id === 'price'){
							$order_value = (int)$order_value;
						}

						if( in_array($field_id, ['price', 'name', 'first_name', 'last_name']) ){
							$entities[$entity][$field_id] = $order_value;
						}else{
							$entities[$entity]['custom_fields_values'][] = [
								'field_id' => (int)$field_id,
								'values' => [
									[
										'value' => $order_value
									]
								]
							];
						}      
					}
					if( $order_info[$order_field] === null){
						$order_info[$order_field] = '';
					}
					$entities['note'] = str_replace('{{'.$order_field.'}}', $order_info[$order_field], $entities['note']);
				}

			}

			$task_data = $settings['task'];
			$entities['task'] = [];
			// создадим задачу
			if( (int)$task_data['status'] ){
				$current_datetime = date('Y-m-d H:i:s');
				$task_complete_till = $task_data['complete_till'];
				$task_times = $task_data['time'];
				$task_complete_till = strtotime($current_datetime . ' +' . $task_complete_till . ' ' . $task_times );

				$entities['task'] = [
					'responsible_user_id' => $responsible_user_id,
					'task_type_id' => (int)$task_data['type'],
					'text' => (string)$task_data['text'],
					'complete_till' => $task_complete_till,
					'duration' =>  1800,
				];
			}
			
			// товары
			$entities['products'] = [];
			$products_total = 0;
			$products_note = ''; 
			if(isset($order_info['products'])){
				$entities['products'] = $order_info['products'];
				if($entities['products']){
					$entities['note'] .= PHP_EOL;
					foreach($entities['products'] as $product){
						if($settings['product_link']){
							$products_note .= $product['name'] . ' x' . $product['quantity'] . ' ' . get_option('home').'?p='.$product['product_id'] . PHP_EOL;
							if(isset($product['options']) and $product['options']){
								foreach($product['options'] as $option){
									$products_note .= '-'.$option. PHP_EOL;
								}
								$entities['note'] .= PHP_EOL;
							}
						}

						$products_total += ($product['price']*$product['quantity']);
					}
				}
			}


			$entities['note'] .= PHP_EOL;
			$entities['note'] .= $order_info['shipping_method'] .': ' . $order_info['shipping_total'] . ' ' . $order_info['currency'] . PHP_EOL;
			$entities['note'] .= $this->lang['payment_method'].': ' . $order_info['payment_method_title'] . PHP_EOL;
			$entities['note'] .= $this->lang['total'].': ' . $order_info['total'] . ' ' . $order_info['currency'] . PHP_EOL . PHP_EOL;
			$entities['note'] .= $products_note; // добавим товары в примечание

			// добавим теги
			foreach(['lead', 'contact', 'company'] as $entity_type){
				if(isset($settings['tags'][$entity_type]) and $settings['tags'][$entity_type]){
					foreach($settings['tags'][$entity_type] as $tag){
						foreach($order_fields as $order_field){
							if( !is_array($order_info[$order_field]) ){
								$tag = str_replace('{{'.$order_field.'}}', $order_info[$order_field], $tag);
							}
						}
						$entities[$entity_type]['_embedded']['tags'][] = ['name' => (string)$tag];
					}
				}
			}

			if(isset($settings['products']) and $settings['products'] == 0){ // товары отключены, обнулим products
				$entities['products'] = [];
			}

			// ответственный контакта
			if(isset($entities['contact'])){
				$entities['contact']['responsible_user_id'] = $responsible_user_id;
			}

			// ответственный компании
			if(isset($entities['company'])){
				$entities['company']['responsible_user_id'] = $responsible_user_id;
			}

			if( isset($settings['antidub']) ){
				$entities['antidub'] = $settings['antidub'];
			}

			return json_encode($entities);
		}

		public function send_order($order_id) {
			// читаем кеш заказов
			$json_path = __DIR__.'/cache_orders.json';
			$cache_orders = [];
			if(file_exists($json_path)){
				$cache_orders = json_decode( file_get_contents($json_path), true );
			}
			if( !in_array($order_id, $cache_orders) ){

				// допишем кеш заказов
				$cache_orders[] = $order_id;
				file_put_contents($json_path, json_encode($cache_orders));

				$result = self::send_data([
					'action' => 'newOrder', 
					'url' => $_SERVER['HTTP_HOST'], 
					'data' => $this->order_data($order_id, false),
					'order_id' => $order_id,
				]);
			}
		}

		public function export(){
			if( !isset($_POST['orders']) or !$_POST['orders'] ){
				return ['status' => 0];
			}
			foreach($_POST['orders'] as $order_id){
				$data = $this->order_data($order_id, false);
				if($data){
					$result = self::send_data([
						'action' => 'newOrder', 
						'url' => $_SERVER['HTTP_HOST'], 
						'data' => $this->order_data($order_id, false),
						'order_id' => $order_id,
					]);
				}

			}
		}

		// Страница настроек
        public function plugin_page(){

			if( isset($_GET['clear_settings']) and (int)$_GET['clear_settings']){ // удалим настройки интеграции
				delete_option(self::PLUGIN_NAME.'_settings');
			}

			if( isset($_GET['clear']) and $_GET['clear'] ){ // обновим данные amocrm
				self::get_amo_fields(1);
			}

			if( isset($_POST['action']) and $_POST['action'] === 'update' and isset($_POST['nonce']) and isset($_POST[self::PLUGIN_NAME]) and wp_verify_nonce( $_POST['nonce'], self::PLUGIN_NAME) ){
				if( !isset($_POST[self::PLUGIN_NAME]['tasks']) ){
					$_POST[self::PLUGIN_NAME]['tasks'] = [];
				}
				if( !isset($_POST[self::PLUGIN_NAME]['tags']['lead']) ){
					$_POST[self::PLUGIN_NAME]['tags']['lead'] = [];
				}
				if( !isset($_POST[self::PLUGIN_NAME]['tags']['contact']) ){
					$_POST[self::PLUGIN_NAME]['tags']['contact'] = [];
				}
				if( !isset($_POST[self::PLUGIN_NAME]['tags']['company']) ){
					$_POST[self::PLUGIN_NAME]['tags']['company'] = [];
				}
				if( !isset($_POST[self::PLUGIN_NAME]['cf']) ){
					$_POST[self::PLUGIN_NAME]['cf'] = [];
				}else{
					foreach($_POST[self::PLUGIN_NAME]['cf'] as $key => $cf_name){
						if($cf_name === ''){
							unset($_POST[self::PLUGIN_NAME]['cf'][$key]);
						}
					}
				}
				
				update_option( self::PLUGIN_NAME.'_settings', $_POST[self::PLUGIN_NAME], 'no' );
			}

			$orders_limit = 10;
			if(isset($_GET['orders_limit'])){
				$orders_limit = (int)$_GET['orders_limit'];
			}

			$widget_code = self::PLUGIN_NAME;
			$order_fields = require_once (__DIR__ . '/templates/order_fields.php');

			$task_default = $this->lang['task_default'];

			$last_cf_id = 0;

			$plugin_settings = get_option(self::PLUGIN_NAME.'_settings');
			if(!$plugin_settings){
				$plugin_settings = [];
			}

			// добавим кастомные поля
			$cf = isset($plugin_settings['cf']) ? $plugin_settings['cf'] : ['_wc_russian_post_tracking_number', '_wc_edostavka_tracking_code', 'boxberry_tracking_number'];
			if($cf){
				foreach($cf as $cf_key => $cf_name){
					$order_fields[] = $cf_name;
					$last_cf_id = $cf_key;
				}
			}		

			$amo_settings = get_option(self::PLUGIN_NAME.'_amocrm');
			if( !$amo_settings ){
				$amo_settings = self::get_amo_fields();
			}
			
			$tab = isset($_GET['tab']) ? $_GET['tab'] : 'tab-header';

			$plugin_settings['plugin_status'] = isset($plugin_settings['plugin_status']) ? (int)$plugin_settings['plugin_status'] : 1;
			$plugin_settings['delay_send_amo'] = isset($plugin_settings['delay_send_amo']) ? (int)$plugin_settings['delay_send_amo'] : 0;
			$plugin_settings['lead_name'] = isset($plugin_settings['lead_name']) ? $plugin_settings['lead_name'] : $this->lang['lead_name_default'];
			$plugin_settings['responsible_list'] = isset($plugin_settings['responsible_list']) ? $plugin_settings['responsible_list'] : [];
			$plugin_settings['pipeline_id'] = isset($plugin_settings['pipeline_id']) ? $plugin_settings['pipeline_id'] : key($amo_settings['pipelines']);
			$plugin_settings['p_status'] = isset($plugin_settings['p_status']) ? $plugin_settings['p_status'] : 0;
		
			$plugin_settings['comparison'] = isset($plugin_settings['comparison']) ? $plugin_settings['comparison'] : [];
			$plugin_settings['product_link'] = isset($plugin_settings['product_link']) ? (int)$plugin_settings['product_link'] : 1;
			$plugin_settings['note_status'] = isset($plugin_settings['note_status']) ? (int)$plugin_settings['note_status'] : 1;
			$plugin_settings['note_template'] = isset($plugin_settings['note_template']) ? $plugin_settings['note_template'] : $this->lang['note_default'];

			$plugin_settings['task_types'] = isset($amo_settings['task_types']) ? $amo_settings['task_types'] : [['id' => 0, 'name' => $this->lang['not_loaded']]];
			$plugin_settings['products'] = isset($plugin_settings['products']) ? $plugin_settings['products'] : [['id' => 0, 'name' => $this->lang['not_loaded']]];
			$plugin_settings['wc_statuses'] = isset($plugin_settings['wc_statuses']) ? $plugin_settings['wc_statuses'] : ['0' => $this->lang['not_loaded']];   

			$task = isset($plugin_settings['task']) ? $plugin_settings['task'] : $task_default;
			$task_types = $amo_settings['task_types'];
			$task_times = $this->lang['times'];

			$plugin_settings['tags']['lead'] = isset($plugin_settings['tags']['lead']) ? $plugin_settings['tags']['lead'] : ['myshop.com'];
			$plugin_settings['tags']['contact'] = isset($plugin_settings['tags']['contact']) ? $plugin_settings['tags']['contact'] : [];
			$plugin_settings['tags']['company'] = isset($plugin_settings['tags']['company']) ? $plugin_settings['tags']['company'] : [];

			$amoFields = isset($amo_settings['fields']) ? $amo_settings['fields'] : ['0' => $this->lang['fields_not_loaded']];
			$amoPipelines = isset($amo_settings['pipelines']) ? $amo_settings['pipelines'] : ['0' => ['name' => $this->lang['pipelines_not_loaded']]];
			$managers = isset($amo_settings['managers']) ? $amo_settings['managers'] : ['0' => $this->lang['managers_not_loaded']];  
			$wc_order_statuses = wc_get_order_statuses();

			$plugin_settings['antidub']['status'] = isset($plugin_settings['antidub']['status']) ? $plugin_settings['antidub']['status'] : 0;
			$ignore = isset($plugin_settings['antidub']['ignore']) ? $plugin_settings['antidub']['ignore'] : [];
			$antidub_fields = isset($amo_settings['antidub_fields']) ? $amo_settings['antidub_fields'] : [0 => $this->lang['not_loaded']];
			$change_responsible = isset($plugin_settings['antidub']['change_responsible']) ? $plugin_settings['antidub']['change_responsible'] : 0;
			$orders = wc_get_orders( array(
				'parent' => 0,
				'limit' => $orders_limit,
				'orderby' => 'date',
				'order' => 'DESC',
			) );

			$title = $this->lang['integration_settings'];
			$version = $amo_settings['version'];
			require_once (__DIR__ . '/templates/settings_page.php');
        }

		// ответственный
		public function responsible_user_id(){
			$plugin_settings = get_option(self::PLUGIN_NAME.'_settings');
			if( !isset($plugin_settings['responsible_list']) ){
				return 0;
			}
		
			$responsible_list = $plugin_settings['responsible_list'];
			$plugin_settings['current_responsible'] = isset($plugin_settings['current_responsible']) ? (int)$plugin_settings['current_responsible'] : 0;
			
			$plugin_settings['current_responsible']++;
			$plugin_settings['current_responsible'] = isset($responsible_list[$plugin_settings['current_responsible']]) ? $plugin_settings['current_responsible'] : 0;
			
			update_option( self::PLUGIN_NAME.'_settings', $plugin_settings );
			return (int)$responsible_list[$plugin_settings['current_responsible']];
		}
			            
        // Меню в настройках
        public function add_admin_menu() {
			add_menu_page(
                $this->lang['plugin_title'], // page title 
                $this->lang['plugin_menu_name'], 
				'manage_options',
				self::PAGE_ID,
				[$this, 'plugin_page'],
				'dashicons-smiley',
				30
			);
        }

		public function orders(){
			if( !isset($_POST['order_id']) or !(int)$_POST['order_id'] ){
				exit(json_encode(['status' => 0]));
			}
			exit(json_encode($this->order($_POST['order_id'])));
		}

		public function order($order_id){
			global $wpdb;
			$order_info = [];
			$order = wc_get_order( $order_id );
			$order_data = $order->get_data(); 

			if($order_data){
				$order_keys = array_keys($order_data);
				foreach($order_keys as $order_key){
					if(!is_array($order_data[$order_key])){
						$key = ($order_key === 'id' ? 'order_id' : $order_key);
						$order_info[$key] = $order_data[$order_key];
					}
				}

				foreach($order_data['billing'] as $billing_key => $billing_value){
					$order_info['billing_'.$billing_key] = $billing_value;
				}

				foreach($order_data['shipping'] as $shipping_key => $shipping_value){
					$order_info['shipping_'.$shipping_key] = $shipping_value;
				}

				foreach($order->get_items() as $order_product){
					$product = $order_product->get_data();
					$product_obj = $order_product->get_product();

					$hidden_order_itemmeta = apply_filters(
						'woocommerce_hidden_order_itemmeta',
						array(
							'_qty',
							'_tax_class',
							'_product_id',
							'_variation_id',
							'_line_subtotal',
							'_line_subtotal_tax',
							'_line_total',
							'_line_tax',
							'method_id',
							'cost',
							'_reduced_stock',
							'_restock_refunded_items',
						)
					);

					$variables = $order_product->get_all_formatted_meta_data( '' );
					if($variables){
						foreach($variables as $variable){
							if ( in_array( $variable->key, $hidden_order_itemmeta, true ) ) {
								continue;
							}
							$product['options'][] = wp_kses_post( $variable->display_key ). ': ' .strip_tags($variable->display_value);
						}
					}

					$price = (int)$product['total'] / $product['quantity'];
					$product['price'] = $price;
					if($product_obj){
						$product['sku'] = $product_obj->get_sku();
					}

					if( isset($product['variation_id']) and $product['variation_id'] and $post_meta_variable = $wpdb->get_results( 'SELECT * FROM '.$wpdb->postmeta.' WHERE post_id = '.$product['variation_id'] ) ){
						foreach($post_meta_variable as $variable){
							if( in_array($variable->meta_key, ['_sku', '_regular_price']) ){
								$product[substr($variable->meta_key, 1)] = $variable->meta_value;
							}
						}
					}
				
					$order_info['products'][] = $product;
				}   
			

				if($order_info['date_created']){
					$order_info['date_created'] = date('d.m.Y H:i:s', strtotime($order_data['date_created']));
				}else{
					$order_info['date_created'] = date('d.m.Y H:i:s');
				}

				if($order_info['date_modified']){
					$order_info['date_modified'] = date('d.m.Y H:i:s', strtotime($order_data['date_modified']));
				}else{
					$order_info['date_modified'] = date('d.m.Y H:i:s');
				}
				
				$order_info['coupon'] = isset($order->get_coupon_codes()['0']) ? implode(', ', $order->get_coupon_codes()) : '';
				$wc_order_statuses = wc_get_order_statuses();
				$order_info['status'] = $wc_order_statuses['wc-'.$order_data['status']];
				$order_info['status_label'] = $order_data['status'];
				$order_info['shipping_method'] = $order->get_shipping_method();

				$order_subtotal = $order->get_subtotal();
				$order_subtotal = number_format( $order_subtotal, 2, '.', '' );
				$order_info['subtotal'] = $order_subtotal;

				// добавим кастомные поля
				$plugin_settings = get_option(self::PLUGIN_NAME.'_settings');
				$cf = isset($plugin_settings['cf']) ? $plugin_settings['cf'] : [];
				if($cf){
					$order_cf_fields = get_post_meta($order_id);
					foreach($cf as $cf_name){
						if( isset($order_cf_fields[$cf_name]) and $order_cf_fields[$cf_name]){ // проверим существования поля
							$order_info[$cf_name] = implode(', ', $order_cf_fields[$cf_name]);
						}else{
							$order_info[$cf_name] = '';
						}
						// ищем в wc_orders_meta заказа (например треки доставок там)
						$order_meta_value = $order->get_meta($cf_name);
						if($order_meta_value){
							$order_info[$cf_name] = $order_meta_value;
						}
					}
				}

				$order_info['customer_total_orders'] = wc_get_customer_order_count( $order_data['customer_id'] );
				$order_info['customer_total_spent'] = wc_get_customer_total_spent( $order_data['customer_id'] );
			}

			return $order_info;
		}

        public static function get_amo_fields($clear_cache = 0){
			$result = self::send_data([
                'action' => 'getAmoCF', 
                'url' => $_SERVER['HTTP_HOST'], 
                'companies' => true, 
                'clear' => $clear_cache,
				'lang' => self::$locale,
            ]);
        	if(isset($result['status']) and $result['status']){
                update_option( self::PLUGIN_NAME.'_amocrm', $result['settings'] );
        	}
        	return isset($result['settings']) ? $result['settings'] : false;
        }

		private static function send_data($data, $attempt = 0){
			$max_attempts = 2;
			$use_proxy = get_option(self::PLUGIN_NAME.'use_proxy', 0); // по умолчанию 0

			$endpoint = $use_proxy ? self::PROXY : self::HOST;
			$query = $data;

			if ($use_proxy) {
				$query['redirect'] = self::HOST;
			}

			error_log('$use_proxy: ' . print_r($use_proxy, true));
			error_log('$endpoint: ' . print_r($endpoint, true));

            $curl = curl_init();
			curl_setopt_array($curl, [
				CURLOPT_RETURNTRANSFER => true,
				CURLOPT_URL => $endpoint,
				CURLOPT_HEADER => false,
				CURLOPT_CUSTOMREQUEST => 'POST',
				CURLOPT_TIMEOUT => self::CURL_TIMEOUT,
				CURLOPT_POSTFIELDS => http_build_query($query),
				CURLOPT_SSL_VERIFYPEER => false,
				CURLOPT_SSL_VERIFYHOST => 2,
			]);
            $json = curl_exec($curl);
			$http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
			$error = curl_error($curl);
            curl_close($curl);

			$attempt++;

			// Успешный запрос
			if (!$error && $http_code >= 200 && $http_code < 300) {
				error_log('Успешный запрос');
				$result = json_decode($json, true);
				error_log('Успешный запрос'.print_r($result, true));
				return $result;
			} else {
				error_log(sprintf(
					'Server Connection Error (Attempt %d/%d, Proxy: %s): %s, HTTP: %d',
					$attempt,
					$max_attempts,
					$use_proxy ? 'Yes' : 'No',
					$error ?: 'No error',
					$http_code
				));
			}

			// Если есть ещё попытки
			if ($attempt < $max_attempts) {
				$new_proxy_setting = !$use_proxy;
				update_option(self::PLUGIN_NAME.'use_proxy', (int)$new_proxy_setting);
				return self::send_data($data, $attempt);
			}

			return false;

		}

        // Активации
        public static function activate() {
            if (!function_exists('curl_version')) {
                deactivate_plugins(plugin_basename(__FILE__), true);
                wp_die(self::$lang_['curl_not_installed']);
            }

			if (!class_exists('WooCommerce')) {
                deactivate_plugins(plugin_basename(__FILE__), true);
                wp_die(self::$lang_['wc_not_installed']);
			}
        
            if( !self::get_amo_fields() ){
                deactivate_plugins(plugin_basename(__FILE__), true);
                wp_die(self::$lang_['site_not_added']);
            }

			global $wpdb;
			$wpdb->query('CREATE TABLE IF NOT EXISTS `'.$wpdb->prefix.'woocommerce_order_utms` (
				`ID` int(11) NOT NULL AUTO_INCREMENT,
				`order_id` int(11) NOT NULL,
				`name` varchar(20) NOT NULL,
				`value` varchar(55) NOT NULL,
				PRIMARY KEY (`ID`)
			)');

			$wpdb->query('CREATE TABLE IF NOT EXISTS `'.$wpdb->prefix.'woocommerce_delayed_orders` (
				ID BIGINT NOT NULL AUTO_INCREMENT,
				order_id BIGINT NOT NULL,
				pushed_at int(11) NOT NULL,
				delay int(11) NOT NULL,
				attempt int(11) NOT NULL,
				PRIMARY KEY (ID)
			)');

			file_put_contents(__DIR__.'/cache_orders.json', json_encode([]));
        }

        // Деактивации
        public static function deactivate() {
			delete_option(self::PLUGIN_NAME.'_amocrm');
			delete_option(self::PLUGIN_NAME.'use_proxy');
        }

		public function translate_name_description($plugins) {
            $plugin_file = plugin_basename(__FILE__);
			if (isset($plugins[$plugin_file])) {
				$plugins[$plugin_file]['Name'] = $this->lang['Name'];
				$plugins[$plugin_file]['Description'] = $this->lang['Description'];
				
			}
            return $plugins;
        }

        // Скрипты и стили
        public function add_scripts( $hook_suffix ) {
            if ( strpos( $hook_suffix, self::PAGE_ID ) === false ) {
                return;
            }
            wp_enqueue_script(self::PLUGIN_NAME.'-chosen-js', plugins_url( '/'.self::PLUGIN_NAME.'/js/chosen.jquery.min.js' ), '', filemtime( __DIR__ . ('/js/chosen.jquery.min.js') ) ,true);
            wp_enqueue_script(self::PLUGIN_NAME.'-js', plugins_url( '/'.self::PLUGIN_NAME.'/js/main.js' ), '', filemtime( __DIR__ . ('/js/main.js') ) ,true);
            
            wp_enqueue_style( self::PLUGIN_NAME.'-chosen-css', plugins_url( '/'.self::PLUGIN_NAME.'/css/chosen.min.css' ), null, filemtime( __DIR__ . ('/css/chosen.min.css') ), 'all' );
            wp_enqueue_style( self::PLUGIN_NAME.'-css', plugins_url( '/'.self::PLUGIN_NAME.'/css/main.css' ), null, filemtime( __DIR__ . ('/css/main.css') ), 'all' );
        }

		public function add_footer(){
		?>   
			<script>
			setTimeout(function() {
				if (typeof Ya !== 'undefined' && Ya._metrika && typeof Ya._metrika.getCounters === 'function') {
					const counters = Ya._metrika.getCounters();
					if (Array.isArray(counters) && counters.length > 0 && counters[0].id) {
						document.cookie = '_ym_counter=' + Ya._metrika.getCounters()[0].id + '; path=/; SameSite=Lax;';
					}
				}
			}, 1000);
			</script>
		<?php 
		}

		public function save_utm_to_cookie() {
			$day = 30;
			$date = time()+3600*24*$day;
			$amo_settings = get_option(self::PLUGIN_NAME.'_amocrm');
			if(isset($amo_settings['utm_fields']) and $amo_settings['utm_fields']){
				foreach($amo_settings['utm_fields'] as $utm){
					if(isset($_GET[$utm])) setcookie($utm, $_GET[$utm], $date, "/"); 
				}
			}
		}
		
    }

    // Инициализация плагина
    $wc_amocrm_instance = new wc_amocrm();

    // Регистрация хуков активации и деактивации
    register_activation_hook(__FILE__, ['wc_amocrm', 'activate']);
    register_deactivation_hook(__FILE__, ['wc_amocrm', 'deactivate']);
}
